/**
 * This file contains the initial script for the hidden window which is connected to
 * the Task Controller servlet. This is a collection of functions used to implement
 * the methods in the TaskControllerQueue class, as well as a global error handler
 * and a heartbeat mechanism to reload the frame should the request be aborted.
 *
 * Please note that several of the function and variable names and values must
 * correspond with names and values in TaskControllerQueue and TaskControllerServlet.
 */


// Simple debugging support.
var debug = false;

function alertIfDebug( msg )
{
  if (debug)
  {
    alert( msg );
  }
}


/**
 * Window scope error handler.  We don't want to show javascript errors
 * to the user, but we don't want to completely ignore them either.  What we'll
 * do is reload the page, passing the error information in the query part of the
 * url.  Not much we can really do to fix things, but the servlet can at least
 * log the problem and we can be sure that the javascript error can't mess up
 * future scripts sent to this frame.
 *
 * @param text
 * @param url
 * @param line
 */
function handleError( text, url, line )
{
  errorInfo =
    "errorText=" + text +
    "&errorURL=" + url +
    "&errorLine=" + line;

  alertIfDebug( errorInfo );

  if ( location.href.indexOf( "?" ) == -1 )
  {
    location.href = location.href + "?" + escape( errorInfo );
  }
  else
  {
    location.href = location.href + "&" + escape( errorInfo );
  }
  return true;
}


// Install the window's error handler.
onerror=handleError;



/**
 * Reload the frame in which this javascript is running.
 *
 * Do not change, remove or rename this function without also changing
 * TaskControllerServlet.RELOAD_SCRIPT, which invokes this function from the servlet.
 */
function reloadHiddenFrame()
{
  alertIfDebug( "about to reload hidden frame" );
  location.reload( true );
}



/**
 * This is the longest period of time that the user might be without tcs
 * input should the connection be lost and not automagically restored by
 * the scripts running in the main window.
 * This should be about 2x-3x the server's heartbeat interval, the default value
 * of which is stored in TaskControllerServlet.DEFAULT_HEARTBEAT_WAIT.
 */
var connectionCheckInterval = 120 * 1000;      // 120 seconds


/**
 * Indicates whether the client browser is still connected to the TaskControllerServlet.
 * Do not change, remove or rename this variable without making the corresponding change
 * to TaskControllerServlet.HEARTBEAT_SCRIPT, which sets this variable from the servlet.
 */
var connectedToServer = true;


/**
 * The following code handles restarting the task controller connection if
 * the pending request should somehow abort.
 *
 * The idea is that every piece of script sent from the server will set
 * connectedToServer to true.  The checkConnection function is set to be
 * called every connectionCheckInterval msecs.  The connectedToServer variable
 * is only set to true by scripts arriving from the server, so if the checkConnection
 * function finds this true, the server must still be there.  The function then
 * sets connectedToServer to be false.  The ratio of the
 * connectionCheckInterval to the heartbeat interval at the server should be set so
 * that new packets arrive more frequently than they are checked.  Therefore, if
 * the checkConnection function ever finds that connectedToServer is still false,
 * the server must not still be connected, so
 * the task controller hidden frame is reloaded.
 */
function checkConnection()
{
  alertIfDebug( "checkConnection, connectedToServer = " + connectedToServer );
  if ( connectedToServer )
  {
    connectedToServer = false;
  }
  else
  {
    reloadHiddenFrame();
  }
}

// Start checking to make sure data is still being received from the pending http request.
setInterval( "checkConnection()", connectionCheckInterval );


/**
 * Open a window.
 *
 * @param url     the url for the new window's content
 * @param name    the name of the new window
 * @param options the JavaScript options string describing the window
 * @param relativeWidth the percentage of the screen used for the new window's width
 * @param relativeHeight the percentage of the screen used for the new window's height
 */
function openWindow( url, name, options, relativeWidth, relativeHeight )
{
  connectedToServer = true;
  if ( relativeWidth > 0 )
  {
    newWidth = window.screen.width * (relativeWidth / 100);
    options += ",width=" + Math.round( newWidth );
  }

  if ( relativeHeight > 0 )
  {
    newHeight = window.screen.height * (relativeHeight / 100);
    options += ",height=" + Math.round( newHeight );
  }
  return window.open( url, name, options )
}

/**
 * Refresh a window.  This means that if the window exists, its contents
 * should be reloaded.  If the window does not exist, it should be
 * created just like in the openWindow method above.
 *
 * @param name    the name of the new window
 * @param url     the url for the new window's content
 * @param options the JavaScript options string describing the window
 * @param relativeWidth the percentage of the screen used for the new window's width
 * @param relativeHeight the percentage of the screen used for the new window's height
 * @param getFocus whether the refreshed window should receive focus
 */
function refreshWindow( name, url, options, relativeWidth, relativeHeight, getFocus )
{
  if ( getFocus == "true" ) {
     theWindow = openWindow( "", name, options, relativeWidth, relativeHeight );
     // If the hmcTaskId property is present, we didn't just open the window,
     // so it's safe just to give it focus.  Otherwise we must reload it.
     if ("hmcTaskId" in theWindow) {
        theWindow.focus();
        return;
     }
  }

  theWindow = openWindow( url, name, options, relativeWidth, relativeHeight );
  if ( getFocus == "true" )
  {
    theWindow.focus();
  }
}

/**
 * This frame should never completely load.  If it does, we'll immediately
 * reload it.
 */
function loaded()
{
  reloadHiddenFrame();
}

/** Install the onload handler. */
onload=loaded;
